// ==++==
//
//   Copyright (c) Microsoft Corporation.  All rights reserved.
//
// ==--==
////////////////////////////////////////////////////////////////////////////
//
//  Class:    TextInfo
//
//  Purpose:  This Class defines behaviors specific to a writing system.
//            A writing system is the collection of scripts and
//            orthographic rules required to represent a language as text.
//
////////////////////////////////////////////////////////////////////////////

namespace System.Globalization
{
    using System.Text;
    using System;
    using System.Runtime.CompilerServices;

    //| <include path='docs/doc[@for="TextInfo"]/*' />
    public class TextInfo
    {
        private TextInfo() {}   // Prevent from being created

        ////////////////////////////////////////////////////////////////////////
        //
        //  ToLower
        //
        //  Converts the character or string to lower case.  Certain locales
        //  have different casing semantics from the file systems in Native.
        //
        ////////////////////////////////////////////////////////////////////////
        //| <include path='docs/doc[@for="TextInfo.ToLower"]/*' />
        public static char ToLower(char c) {
            return GetLowerUpperCase(lowerCaseTable, c);
        }
        //| <include path='docs/doc[@for="TextInfo.ToLower1"]/*' />
        public static String ToLower(String str) {
            if (str == null) {
                throw new ArgumentNullException("str");
            }
            if (str.Length == 0) {
                return str;
            }
            char[] chars = str.ToCharArray();
            int length = chars.Length;
            for (int i = 0; i < length; i++) {
                chars[i] = GetLowerUpperCase(lowerCaseTable, chars[i]);
            }
            return String.StringCTOR(chars, 0, length);
        }

        internal static int GetCaseInsensitiveHashCode(String str) {
            String upperString = ToUpper(str);
            return upperString.GetHashCode();
        }

        ////////////////////////////////////////////////////////////////////////
        //
        //  ToUpper
        //
        //  Converts the character or string to upper case.  Certain locales
        //  have different casing semantics from the file systems in Native.
        //
        ////////////////////////////////////////////////////////////////////////
        //| <include path='docs/doc[@for="TextInfo.ToUpper"]/*' />
        public static char ToUpper(char c) {
            return GetLowerUpperCase(upperCaseTable, c);
        }

        public static String ToUpper(String str) {
            if (str == null) {
                throw new ArgumentNullException("str");
            }
            if (str.Length == 0) {
                return str;
            }
            char[] chars = str.ToCharArray();
            int length = chars.Length;
            for (int i = 0; i < length; i++) {
                chars[i] = GetLowerUpperCase(upperCaseTable, chars[i]);
            }
            return String.StringCTOR(chars, 0, length);
        }

        private static char GetLowerUpperCase(short[] table, char c) {
            short firstIndex = table[c >> 8];
            short secondIndex = table[firstIndex + ((c >> 4) & 0xf)];
            return unchecked((char) (c + table[secondIndex + (c & 0xf)]));
        }

        private static bool IsLetterCategory(UnicodeCategory uc) {
            return (uc == UnicodeCategory.UppercaseLetter
                    || uc == UnicodeCategory.LowercaseLetter
                    || uc == UnicodeCategory.TitlecaseLetter
                    || uc == UnicodeCategory.ModifierLetter
                    || uc == UnicodeCategory.OtherLetter);
        }

        private static short[] upperCaseTable = new short[1152] {
            // firstIndex [0..255]
            256 ,272 ,288 ,304 ,320 ,336 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 , // 0
            352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,368 ,384 , // 16
            352 ,400 ,352 ,352 ,416 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 , // 32
            352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 , // 48
            352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 , // 64
            352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 , // 80
            352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 , // 96
            352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 , // 112
            352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 , // 128
            352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 , // 144
            352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 , // 160
            352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 , // 176
            352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 , // 192
            352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 , // 208
            352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 , // 224
            352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,432 , // 240
            // secondIndex [256..447]
            448 ,448 ,448 ,448 ,448 ,448 ,464 ,480 ,448 ,448 ,448 ,448 ,448 ,448 ,496 ,512 , // 256
            528 ,528 ,528 ,544 ,560 ,528 ,528 ,576 ,592 ,608 ,624 ,640 ,656 ,672 ,528 ,688 , // 272
            528 ,704 ,448 ,448 ,448 ,720 ,736 ,752 ,768 ,784 ,448 ,448 ,448 ,448 ,448 ,448 , // 288
            448 ,448 ,448 ,448 ,448 ,448 ,448 ,448 ,448 ,448 ,800 ,464 ,816 ,448 ,832 ,448 , // 304
            448 ,448 ,448 ,496 ,496 ,848 ,528 ,528 ,864 ,528 ,528 ,528 ,880 ,528 ,896 ,912 , // 320
            448 ,448 ,448 ,448 ,448 ,448 ,928 ,944 ,960 ,448 ,448 ,448 ,448 ,448 ,448 ,448 , // 336
            448 ,448 ,448 ,448 ,448 ,448 ,448 ,448 ,448 ,448 ,448 ,448 ,448 ,448 ,448 ,448 , // 352
            528 ,528 ,528 ,528 ,528 ,528 ,528 ,528 ,528 ,976 ,528 ,528 ,528 ,528 ,528 ,992 , // 368
            1008,1024,1008,1008,1024,1040,1008,1056,448 ,448 ,448 ,1072,448 ,1072,1088,448 , // 384
            448 ,448 ,448 ,448 ,448 ,448 ,448 ,1104,448 ,448 ,448 ,448 ,448 ,448 ,448 ,448 , // 400
            448 ,448 ,448 ,448 ,448 ,448 ,448 ,448 ,448 ,448 ,448 ,448 ,448 ,1120,1136,448 , // 416
            448 ,448 ,448 ,448 ,464 ,480 ,448 ,448 ,448 ,448 ,448 ,448 ,448 ,448 ,448 ,448 , // 432
            // adjustments [448..1151]
            0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   , // 448
            0   ,-32 ,-32 ,-32 ,-32 ,-32 ,-32 ,-32 ,-32 ,-32 ,-32 ,-32 ,-32 ,-32 ,-32 ,-32 , // 464
            -32 ,-32 ,-32 ,-32 ,-32 ,-32 ,-32 ,-32 ,-32 ,-32 ,-32 ,0   ,0   ,0   ,0   ,0   , // 480
            -32 ,-32 ,-32 ,-32 ,-32 ,-32 ,-32 ,-32 ,-32 ,-32 ,-32 ,-32 ,-32 ,-32 ,-32 ,-32 , // 496
            -32 ,-32 ,-32 ,-32 ,-32 ,-32 ,-32 ,0   ,-32 ,-32 ,-32 ,-32 ,-32 ,-32 ,-32 ,121 , // 512
            0   ,-1  ,0   ,-1  ,0   ,-1  ,0   ,-1  ,0   ,-1  ,0   ,-1  ,0   ,-1  ,0   ,-1  , // 528
            0   ,0   ,0   ,-1  ,0   ,-1  ,0   ,-1  ,0   ,0   ,-1  ,0   ,-1  ,0   ,-1  ,0   , // 544
            -1  ,0   ,-1  ,0   ,-1  ,0   ,-1  ,0   ,-1  ,0   ,0   ,-1  ,0   ,-1  ,0   ,-1  , // 560
            0   ,-1  ,0   ,-1  ,0   ,-1  ,0   ,-1  ,0   ,0   ,-1  ,0   ,-1  ,0   ,-1  ,0   , // 576
            0   ,0   ,0   ,-1  ,0   ,-1  ,0   ,0   ,-1  ,0   ,0   ,0   ,-1  ,0   ,0   ,0   , // 592
            0   ,0   ,-1  ,0   ,0   ,0   ,0   ,0   ,0   ,-1  ,0   ,0   ,0   ,0   ,0   ,0   , // 608
            0   ,-1  ,0   ,-1  ,0   ,-1  ,0   ,0   ,-1  ,0   ,0   ,0   ,0   ,-1  ,0   ,0   , // 624
            -1  ,0   ,0   ,0   ,-1  ,0   ,-1  ,0   ,0   ,-1  ,0   ,0   ,0   ,-1  ,0   ,0   , // 640
            0   ,0   ,0   ,0   ,0   ,0   ,-2  ,0   ,0   ,-2  ,0   ,0   ,-2  ,0   ,-1  ,0   , // 656
            -1  ,0   ,-1  ,0   ,-1  ,0   ,-1  ,0   ,-1  ,0   ,-1  ,0   ,-1  ,-79 ,0   ,-1  , // 672
            0   ,0   ,0   ,-2  ,0   ,-1  ,0   ,0   ,0   ,0   ,0   ,-1  ,0   ,-1  ,0   ,-1  , // 688
            0   ,-1  ,0   ,-1  ,0   ,-1  ,0   ,-1  ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   , // 704
            0   ,0   ,0   ,-210,-206,0   ,-205,-205,0   ,-202,0   ,-203,0   ,0   ,0   ,0   , // 720
            -205,0   ,0   ,-207,0   ,0   ,0   ,0   ,-209,-211,0   ,0   ,0   ,0   ,0   ,-211, // 736
            0   ,0   ,-213,0   ,0   ,-214,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   , // 752
            0   ,0   ,0   ,-218,0   ,0   ,0   ,0   ,-218,0   ,-217,-217,0   ,0   ,0   ,0   , // 768
            0   ,0   ,-219,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   , // 784
            0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,-38 ,-37 ,-37 ,-37 , // 800
            -32 ,-32 ,-31 ,-32 ,-32 ,-32 ,-32 ,-32 ,-32 ,-32 ,-32 ,-32 ,-64 ,-63 ,-63 ,0   , // 816
            0   ,0   ,0   ,-1  ,0   ,-1  ,0   ,-1  ,0   ,-1  ,0   ,-1  ,0   ,-1  ,0   ,-1  , // 832
            0   ,-80 ,-80 ,-80 ,-80 ,-80 ,-80 ,-80 ,-80 ,-80 ,-80 ,-80 ,-80 ,0   ,-80 ,-80 , // 848
            0   ,-1  ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   , // 864
            0   ,0   ,-1  ,0   ,-1  ,0   ,0   ,0   ,-1  ,0   ,0   ,0   ,-1  ,0   ,0   ,0   , // 880
            0   ,-1  ,0   ,-1  ,0   ,-1  ,0   ,-1  ,0   ,-1  ,0   ,-1  ,0   ,0   ,0   ,-1  , // 896
            0   ,-1  ,0   ,-1  ,0   ,-1  ,0   ,0   ,0   ,-1  ,0   ,0   ,0   ,0   ,0   ,0   , // 912
            0   ,-48 ,-48 ,-48 ,-48 ,-48 ,-48 ,-48 ,-48 ,-48 ,-48 ,-48 ,-48 ,-48 ,-48 ,-48 , // 928
            -48 ,-48 ,-48 ,-48 ,-48 ,-48 ,-48 ,-48 ,-48 ,-48 ,-48 ,-48 ,-48 ,-48 ,-48 ,-48 , // 944
            -48 ,-48 ,-48 ,-48 ,-48 ,-48 ,-48 ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   , // 960
            0   ,-1  ,0   ,-1  ,0   ,-1  ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   , // 976
            0   ,-1  ,0   ,-1  ,0   ,-1  ,0   ,-1  ,0   ,-1  ,0   ,0   ,0   ,0   ,0   ,0   , // 992
            8   ,8   ,8   ,8   ,8   ,8   ,8   ,8   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   , // 1008
            8   ,8   ,8   ,8   ,8   ,8   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   , // 1024
            0   ,8   ,0   ,8   ,0   ,8   ,0   ,8   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   , // 1040
            74  ,74  ,86  ,86  ,86  ,86  ,100 ,100 ,128 ,128 ,112 ,112 ,126 ,126 ,0   ,0   , // 1056
            8   ,8   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   , // 1072
            8   ,8   ,0   ,0   ,0   ,7   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   , // 1088
            -16 ,-16 ,-16 ,-16 ,-16 ,-16 ,-16 ,-16 ,-16 ,-16 ,-16 ,-16 ,-16 ,-16 ,-16 ,-16 , // 1104
            -26 ,-26 ,-26 ,-26 ,-26 ,-26 ,-26 ,-26 ,-26 ,-26 ,-26 ,-26 ,-26 ,-26 ,-26 ,-26 , // 1120
            -26 ,-26 ,-26 ,-26 ,-26 ,-26 ,-26 ,-26 ,-26 ,-26 ,0   ,0   ,0   ,0   ,0   ,0   , // 1136
        };

        private static short[] lowerCaseTable = new short[1136] {
            // firstIndex [0..255]
            256 ,272 ,288 ,304 ,320 ,336 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 , // 0
            368 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,384 ,400 , // 16
            352 ,416 ,352 ,352 ,432 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 , // 32
            352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 , // 48
            352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 , // 64
            352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 , // 80
            352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 , // 96
            352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 , // 112
            352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 , // 128
            352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 , // 144
            352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 , // 160
            352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 , // 176
            352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 , // 192
            352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 , // 208
            352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 , // 224
            352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,352 ,448 , // 240
            // secondIndex [256..463]
            464 ,464 ,464 ,464 ,480 ,496 ,464 ,464 ,464 ,464 ,464 ,464 ,512 ,528 ,464 ,464 , // 256
            544 ,544 ,544 ,560 ,576 ,544 ,544 ,592 ,608 ,624 ,640 ,656 ,672 ,688 ,544 ,704 , // 272
            544 ,720 ,464 ,464 ,464 ,464 ,464 ,464 ,464 ,464 ,464 ,464 ,464 ,464 ,464 ,464 , // 288
            464 ,464 ,464 ,464 ,464 ,464 ,464 ,464 ,736 ,480 ,752 ,464 ,464 ,464 ,768 ,464 , // 304
            784 ,512 ,512 ,464 ,464 ,464 ,544 ,544 ,800 ,544 ,544 ,544 ,816 ,544 ,832 ,848 , // 320
            464 ,464 ,464 ,864 ,880 ,896 ,464 ,464 ,464 ,464 ,464 ,464 ,464 ,464 ,464 ,464 , // 336
            464 ,464 ,464 ,464 ,464 ,464 ,464 ,464 ,464 ,464 ,464 ,464 ,464 ,464 ,464 ,464 , // 352
            464 ,464 ,464 ,464 ,464 ,464 ,464 ,464 ,464 ,464 ,880 ,880 ,912 ,464 ,464 ,464 , // 368
            544 ,544 ,544 ,544 ,544 ,544 ,544 ,544 ,544 ,928 ,544 ,544 ,544 ,544 ,544 ,944 , // 384
            960 ,976 ,960 ,960 ,976 ,992 ,960 ,464 ,464 ,464 ,464 ,1008,1024,1040,1056,1072, // 400
            464 ,464 ,464 ,464 ,464 ,464 ,1088,464 ,464 ,464 ,464 ,464 ,464 ,464 ,464 ,464 , // 416
            464 ,464 ,464 ,464 ,464 ,464 ,464 ,464 ,464 ,464 ,464 ,1104,1120,464 ,464 ,464 , // 432
            464 ,464 ,480 ,496 ,464 ,464 ,464 ,464 ,464 ,464 ,464 ,464 ,464 ,464 ,464 ,464 , // 448
            // adjustments [464..1135]
            0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   , // 464
            0   ,32  ,32  ,32  ,32  ,32  ,32  ,32  ,32  ,32  ,32  ,32  ,32  ,32  ,32  ,32  , // 480
            32  ,32  ,32  ,32  ,32  ,32  ,32  ,32  ,32  ,32  ,32  ,0   ,0   ,0   ,0   ,0   , // 496
            32  ,32  ,32  ,32  ,32  ,32  ,32  ,32  ,32  ,32  ,32  ,32  ,32  ,32  ,32  ,32  , // 512
            32  ,32  ,32  ,32  ,32  ,32  ,32  ,0   ,32  ,32  ,32  ,32  ,32  ,32  ,32  ,0   , // 528
            1   ,0   ,1   ,0   ,1   ,0   ,1   ,0   ,1   ,0   ,1   ,0   ,1   ,0   ,1   ,0   , // 544
            0   ,0   ,1   ,0   ,1   ,0   ,1   ,0   ,0   ,1   ,0   ,1   ,0   ,1   ,0   ,1   , // 560
            0   ,1   ,0   ,1   ,0   ,1   ,0   ,1   ,0   ,0   ,1   ,0   ,1   ,0   ,1   ,0   , // 576
            1   ,0   ,1   ,0   ,1   ,0   ,1   ,0   ,-121,1   ,0   ,1   ,0   ,1   ,0   ,0   , // 592
            0   ,210 ,1   ,0   ,1   ,0   ,206 ,1   ,0   ,205 ,205 ,1   ,0   ,0   ,79  ,202 , // 608
            203 ,1   ,0   ,205 ,207 ,0   ,211 ,209 ,1   ,0   ,0   ,0   ,211 ,213 ,0   ,214 , // 624
            1   ,0   ,1   ,0   ,1   ,0   ,0   ,1   ,0   ,218 ,0   ,0   ,1   ,0   ,218 ,1   , // 640
            0   ,217 ,217 ,1   ,0   ,1   ,0   ,219 ,1   ,0   ,0   ,0   ,1   ,0   ,0   ,0   , // 656
            0   ,0   ,0   ,0   ,2   ,0   ,0   ,2   ,0   ,0   ,2   ,0   ,0   ,1   ,0   ,1   , // 672
            0   ,1   ,0   ,1   ,0   ,1   ,0   ,1   ,0   ,1   ,0   ,1   ,0   ,0   ,1   ,0   , // 688
            0   ,2   ,0   ,0   ,1   ,0   ,0   ,0   ,0   ,0   ,1   ,0   ,1   ,0   ,1   ,0   , // 704
            1   ,0   ,1   ,0   ,1   ,0   ,1   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   , // 720
            0   ,0   ,0   ,0   ,0   ,0   ,38  ,0   ,37  ,37  ,37  ,0   ,64  ,0   ,63  ,63  , // 736
            32  ,32  ,0   ,32  ,32  ,32  ,32  ,32  ,32  ,32  ,32  ,32  ,0   ,0   ,0   ,0   , // 752
            0   ,0   ,1   ,0   ,1   ,0   ,1   ,0   ,1   ,0   ,1   ,0   ,1   ,0   ,1   ,0   , // 768
            0   ,80  ,80  ,80  ,80  ,80  ,80  ,80  ,80  ,80  ,80  ,80  ,80  ,0   ,80  ,80  , // 784
            1   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   , // 800
            0   ,1   ,0   ,1   ,0   ,0   ,0   ,1   ,0   ,0   ,0   ,1   ,0   ,0   ,0   ,0   , // 816
            1   ,0   ,1   ,0   ,1   ,0   ,1   ,0   ,1   ,0   ,1   ,0   ,0   ,0   ,1   ,0   , // 832
            1   ,0   ,1   ,0   ,1   ,0   ,0   ,0   ,1   ,0   ,0   ,0   ,0   ,0   ,0   ,0   , // 848
            0   ,48  ,48  ,48  ,48  ,48  ,48  ,48  ,48  ,48  ,48  ,48  ,48  ,48  ,48  ,48  , // 864
            48  ,48  ,48  ,48  ,48  ,48  ,48  ,48  ,48  ,48  ,48  ,48  ,48  ,48  ,48  ,48  , // 880
            48  ,48  ,48  ,48  ,48  ,48  ,48  ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   , // 896
            48  ,48  ,48  ,48  ,48  ,48  ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   , // 912
            1   ,0   ,1   ,0   ,1   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   , // 928
            1   ,0   ,1   ,0   ,1   ,0   ,1   ,0   ,1   ,0   ,0   ,0   ,0   ,0   ,0   ,0   , // 944
            0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,-8  ,-8  ,-8  ,-8  ,-8  ,-8  ,-8  ,-8  , // 960
            0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,-8  ,-8  ,-8  ,-8  ,-8  ,-8  ,0   ,0   , // 976
            0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,-8  ,0   ,-8  ,0   ,-8  ,0   ,-8  , // 992
            0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,-8  ,-8  ,-74 ,-74 ,0   ,0   ,0   ,0   , // 1008
            0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,-86 ,-86 ,-86 ,-86 ,0   ,0   ,0   ,0   , // 1024
            0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,-8  ,-8  ,-100,-100,0   ,0   ,0   ,0   , // 1040
            0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,-8  ,-8  ,-112,-112,-7  ,0   ,0   ,0   , // 1056
            0   ,0   ,0   ,0   ,0   ,0   ,0   ,0   ,-128,-128,-126,-126,0   ,0   ,0   ,0   , // 1072
            16  ,16  ,16  ,16  ,16  ,16  ,16  ,16  ,16  ,16  ,16  ,16  ,16  ,16  ,16  ,16  , // 1088
            0   ,0   ,0   ,0   ,0   ,0   ,26  ,26  ,26  ,26  ,26  ,26  ,26  ,26  ,26  ,26  , // 1104
            26  ,26  ,26  ,26  ,26  ,26  ,26  ,26  ,26  ,26  ,26  ,26  ,26  ,26  ,26  ,26  , // 1120
        };
    }
}
